sap.ui.define(["sap/base/Log"], function (Log) {
	"use strict";

	return {
		/**
		 * Create a instance of Nubexx or SAP PDFViewer control.
		 *
		 * @param {string} type                    - Pass "nbx" to load Nubexx PDFViewer control
		 * @returns {Promise<sap.ui.core.Control>} - Promise mit dem geladenen Control.
		 */
		create: function (type) {
			return new Promise(function (resolve, reject) {
				const aRequiredModules = type === "nbx" ?
					["com/nbx/pdfviewer/PDFViewer", "com/nbx/pdfviewer/Option"] :
					["sap/m/PDFViewer"];

				sap.ui.require(aRequiredModules, function (...aLoadedModules) {
					if (!aLoadedModules) {
						return reject("Required modules could not be loaded:", aLoadedModules);
					}

					const [PdfViewer, Option] = aLoadedModules;
					return resolve({ PdfViewer, Option });
				});
			});
		},

		/**
		 * Replace oControlToReplace width oNewControl and transfer selected properties
		 * _copyAggregations and  _copyBindings;
		 *
		 * @param {sap.ui.core.Control} oControlToReplace   - Control to replace
		 * @param {sap.ui.core.Control} oNewControl         - New control
		 */
		replaceControl: function (oControlToReplace, oNewControl) {
			const aIgnoreProperties = ["showPrintButton", "showToolbar", "showWatermark", "downloadFileName"];
			const aIgnoreAggregations = ["pdfViewerOptions"];

			if (!oControlToReplace) {
				Log.error("Control to replace could not be found!");
				return;
			}

			this._copyProperties(oControlToReplace, oNewControl, aIgnoreProperties);
			oNewControl.sId = oControlToReplace.getId();

			this._copyAggregations(oControlToReplace, oNewControl, aIgnoreAggregations);
			this._copyBindings(oControlToReplace, oNewControl);

			const oParent = oControlToReplace.getParent();
			const iIndex = oParent.getItems().indexOf(oControlToReplace);

			oParent.removeItem(oControlToReplace);
			oControlToReplace.destroy();
			oParent.insertItem(oNewControl, iIndex);
		},

		/**
		 * Copy control properties with exceptions.
		 *
		 * @param {sap.ui.core.Control} oSource - Source control
		 * @param {sap.ui.core.Control} oTarget - Target control
		 * @param {Array} ignoreProperties		- Array with properties to ignore
		 */
		_copyProperties: function (oSource, oTarget, ignoreProperties) {
			for (const sProperty of Object.keys(oSource.getMetadata().getAllProperties())) {
				if (ignoreProperties.indexOf(sProperty) === -1) {
					oTarget.setProperty(sProperty, oSource.getProperty(sProperty));
				}
			}
		},

		/**
		 * Copy control aggregations.
		 *
		 * @param {sap.ui.core.Control} oSource - Source control
		 * @param {sap.ui.core.Control} oTarget - Target control
		 * @param {Array} ignoreAggregations    - Array with aggregation names to ignore
		 */
		_copyAggregations: function (oSource, oTarget, ignoreAggregations) {
			const aAggregations = oSource.getMetadata().getAllAggregations();
			Object.keys(aAggregations).forEach(function (aggregationName) {
				if (ignoreAggregations.indexOf(aggregationName) === -1) {
					const vAggregation = oSource.getAggregation(aggregationName);
					if (Array.isArray(vAggregation)) {
						vAggregation.forEach(function (item) {
							oTarget.addAggregation(aggregationName, item);
						});
					} else if (vAggregation) {
						oTarget.setAggregation(aggregationName, vAggregation);
					}
				}
			});
		},

		/**
		 * Copy control bindings.
		 *
		 * @param {sap.ui.core.Control} oSource - Source control
		 * @param {sap.ui.core.Control} oTarget - Target control
		 */
		_copyBindings: function (oSource, oTarget) {
			const oBindingInfos = oSource.mBindingInfos;

			Object.keys(oBindingInfos).forEach(function (propertyName) {
				const oBindingInfo = oBindingInfos[propertyName];
				if (oBindingInfo.parts) {
					oTarget.bindProperty(propertyName, {
						parts: oBindingInfo.parts,
						formatter: oBindingInfo.formatter,
						mode: oBindingInfo.mode
					});
				} else {
					// single binding
					oTarget.bindProperty(propertyName, {
						path: oBindingInfo.path,
						model: oBindingInfo.model,
						type: oBindingInfo.type,
						formatter: oBindingInfo.formatter,
						mode: oBindingInfo.mode
					});
				}
			});
		}
	};
});
